unit DirList;
{
    UNIT DirList
    Version number 1.0(out of beta)

This unit contains the FTP dirlist manager TDirList class.
TDirList class is responsible for providing information from a text directory
list.

Created by Pter Karsai }

interface

uses SysUtils, WinTypes, WinProcs, Messages, Classes, Tokenizer, Forms;

const MaxDirLineLength = 514; { 512 + 2 }
{ Constant MaxDirLineLenght limits the line length in the received
  server directory list. Because every line stored an PChar, the
  real character limit is MaxDirLineLength - 1 'cause of the #0
  terminator character.
  I had to use PChar instead of the String, because I don't know
  any limitation for the line length given by the 'ls -lGa' unix command,
  maybe the filename has some for 255 characters but a line contains
  a lot of additional information, e.g. access rights, owner, size
  and so on }

{ This implementation does *not* store the directory information in
  itself, just think: max. 1024 files stored in one time requires
  1024 pointers, 1024 * 4 bytes and if MaxDirLineLength = 512, it requires
  1024 * 512 bytes, too. It's 528384 bytes, quite a lot, isn't it? }

      MaxFileLines = 4096;  { set to 4096, 11th of September, 99.}

{ Constant MaxFileLines limits the maximal line count in the directory-
  info file. Since searching in a file isn't a kind of Speedy Gonzales-like
  action, TDirList class builds an internal table to determine line starts
  in the directory info file. Be careful set this constant higher, it needs
  MaxFileLines * 4 bytes! Internal table: [private] LStartTable. }

type
{------------------------------------------------------------------------------}
{ Record types }
  DirDetailRecord = record
  { frequently used info }
    dd_Name       : string;     { file or directory name }
    dd_Size       : longint;    { file or directory size in bytes }
    dd_Date       : string[16]; { system-specific date, e.g. 25 May 17:26 }
  { additional }
    dd_Rights     : string[11]; { file or directory access rights }
    dd_Owner      : string[16]; { owner of the file or directory }
    dd_OwnerGroup : string[16]; { user group of the owner }
    dd_INodes     : word;       { number of iNodes, used by Unix systems }
  { packed info }
    dd_Info       : char;       { F=file, D=directory, L=link, U=unknown }
    dd_LinksTo    : string;     { if dd_Info=L, it contains the link }
  end;

{ DirDetailRecord type stores information about a directory entry, you'll use
  this type as returning value of function GetDetails() }

{------------------------------------------------------------------------------}
{ Exceptions }

  EInvalidFormat = class(Exception);
{ Thrown if the directory info format is different than Unix ls -lGa,
  or thrown on every weekend when you stay at home instead of walking
  outside in the nice night or spending your time in a cheap pub :-))) }

  ELineTooLong   = class(Exception);
{ OooOooOoo, I'm fed up with the error handling! Well, ELineTooLong exception
  will be thrown... guess... congratulations! you got it! it will be thrown
  when a line is too long in the file }

{------------------------------------------------------------------------------}
{ And here's... the TDirList class! }

  TDirList = class(TObject)
  private
  { internal data }
    LineCount   : word;  { lines in the directory info file }
    LStartTable : array [0..MaxFileLines - 1] of longint; { file pointer table }
    DirFile     : file;  { untyped file handle, assigned to directory info
                           file }
    NoIOError   : boolean; { determines the I/O state. If NoIOError = FALSE,
                             none of the routines will work. }

  { private methods }
    function BuildTable: word;
{ BuildTable() is for internal use only; it fills 'LStartTable' with starting
  positions of the lines. The returning value gives the number of lines read
  to table. }

  public

{ ----------------------------------------------------------------------------}
{ file manager methods }
{ ----------------------------------------------------------------------------}
    function ReadLongLine(LineNo: word; var LongLine: PChar): boolean;
{ ReadLongLine() read the LineNo'th line into variable parameter 'LongLine'.
  It's your responsibilty to allocate LongLine before call ReadLongLine().
  The numbering of the lines start at 0.
  Returns FALSE if no 'LineNo' line number is in the internal table.
  Throws (passes) EInOutError on any I/O error}

    function GetLineCount: word;
{ GetLineCount() returns with the number of lines read by BuildTable() }

{ ----------------------------------------------------------------------------}
{ directory information managert methods }
{ ----------------------------------------------------------------------------}
    function GetDetails(LineNo: word; var DirRec: DirDetailRecord): boolean;
{ GetDetails() returns with a detailed info about a directory entry in variable
  parameter 'DirRec'.
  It uses ReadLongLine(), so you'll get all the exceptions ReadLongLine()
  can throw. Returns FALSE if no 'LineNo' line number is in the internal table.}

{ ----------------------------------------------------------------------------}
{ constructor and destructor }
{ ----------------------------------------------------------------------------}
    constructor Create(DirFileName: string);
{ DirFileName parameter defines the file stores the directory information.
  You would *not* modify or delete this file while TDirList instanted,
  anyway you'll get incorrect info by this object or crash your
  application, 'cause it doesn't store the file data in itself, it
  read the file when need to do it!
  Throws EInOutError on any I/O error.
  Throws (passes) EInvalidFormat if no info-lines found.
  Throws (passes) ELineTooLong if a line is too long to read - increment
  MaxDirLineLength. }

    destructor Destroy; override;
    procedure Free;
end;


implementation

{------------------------------------------------------------------------------}
{------------------------------------------------------------------------------}
{------ file manager methods --------------------------------------------------}
{------------------------------------------------------------------------------}
{------------------------------------------------------------------------------}
function TDirList.BuildTable: word;
var xFilePos    : longint;     { Mulder and Scully helped me a lot... Thx :) }
    q           : PChar;       { q... q... qhyy ahhpsi! }
    NumRead     : integer;     { avoid EOF problems with BlockRead,
                                 see Help/BlockRead }
    CurrentLine : PChar;       { buffer }
begin
{ allocate buffer }
     CurrentLine:= StrAlloc(MaxDirLineLength);
{ at first... no lines read }
     Result:= 0;
     xFilePos:= 0; { we start at the beginning of the file }

{ read all the lines until EOF, if no I/O errors appear }
     while not Eof(DirFile) and NoIOError  do begin
     { stop reading and reraise exceptions on any I/O exceptions }
           try
              BlockRead(DirFile, CurrentLine[0], MaxDirLineLength - 1, NumRead);
           except
              on EInOutError do begin
                 NoIOError:= false;
                 StrDispose(CurrentLine);
                 raise
              end
           end;

     { well it's ok, there was no error... }
           q:= StrPos(CurrentLine, #10);
           if q = nil then begin
           { ... but if the line is too long... }
              NoIOError:= false;
              StrDispose(CurrentLine);
              raise ELineTooLong.Create('Line was too long! '+IntToStr(Result));
           end
     { ... and well, we hope, it was *not* too long }
           else begin
              q[0]:= #0;  { now our string looks like [blah...blah#13#0] }
           { ok, ok but is it an information line or something like
             'total 14824'? We can determine, the 9th- token of a legal
             directory info line is the filename. Are there 9 token? }

              if TTokenizer.GetStringTokenStart(9, #32, CurrentLine) <> nil then
              begin
                 LStartTable[Result]:= xFilePos; { store entry }
                 inc(Result);
              end;

              xFilePos:= xFilePos + q - CurrentLine + 1; {...simple...}
              Seek(DirFile, xFilePos); { ok, I don't handle exceptions, 'cause
                                         I can just seek to EOF. If you don't
                                         believe me - The truth is out there! }
           end; { line length checker IF }
     end; { read cycle }

{ free buffer }
     StrDispose(CurrentLine);
end;

{------------------------------------------------------------------------------}

function TDirList.ReadLongLine(LineNo: word; var LongLine: PChar): boolean;
var CurrentLine: PChar;   { buffer }
    NumRead    : integer; { as in the BuildTable() }
    q          : PChar;   { I can't explain... it's such expressive, not? :-)}
begin
{ set returning value }
     Result:= LineNo <= LineCount;
{ there's such line, do its work!... }
     if Result then begin
     { allocate buffer }
       CurrentLine:= StrAlloc(MaxDirLineLength);
     { try to read }
       try
           Seek(DirFile, LStartTable[LineNo]);
           BlockRead(DirFile, CurrentLine[0], MaxDirLineLength - 1, NumRead);
       except
           on EInOutError do begin
              StrDispose(CurrentLine); NoIOError:= false; raise end
       end;

     { ok, search... }
       q:= StrPos(CurrentLine, #13); { string end looks like #13#10.. I hope. }
     { ...and copy }
       StrLCopy(LongLine, CurrentLine, q - CurrentLine);

     { free buffer }
       StrDispose(CurrentLine);
     end;
end;
{------------------------------------------------------------------------------}

function TDirList.GetLineCount: word;
begin
     Result:= LineCount;
end;

{------------------------------------------------------------------------------}
{------------------------------------------------------------------------------}
{------ directory infomation manager methods ----------------------------------}
{------------------------------------------------------------------------------}
{------------------------------------------------------------------------------}
{ The directory information by the FTP LIST command on the most of the systems
  looks like a list of a Unix 'ls -lgA' command... on the most of... :(}

function TDirList.GetDetails(LineNo: word; var DirRec: DirDetailRecord): boolean;
var temp    : PChar; { yet another boring temporary variable }
    dirLine : PChar; { stores the requested directory info line }
begin
{ read the required line... }
     dirLine:= StrAlloc(MaxDirLineLength);
     if not ReadLongLine(LineNo, dirLine) then
     begin
        StrDispose(dirLine);
        Result:= false;
        Exit;
     end;

{ allocate temporary buffer }
     temp:= StrAlloc(256);  { 256 byte ought be enough for everybody... hm :) }

{ my fingers get tired... I never used 'with' clause before :) }
     with DirRec do begin
     { get the file name - we don't care NOW if it's a link }
          { there must be a 9th token, else exception generated in BuildTable()}
          dd_Name:= StrPas(TTokenizer.GetStringTokenStart(9, #32, dirLine));
     { size... }
          TTokenizer.GetStringToken(5, #32, dirLine, temp);
          try
             dd_Size:= StrToInt(StrPas(temp));
          except
             on EConvertError do dd_size:= -1; { oh don't be such a fool :) }
             { i'll be... ;-)))}
          end;
     { date... }
          TTokenizer.GetStringToken(6, #32, dirLine, temp);
            dd_Date:= StrPas(temp);
          TTokenizer.GetStringToken(7, #32, dirLine, temp);
            dd_Date:= dd_Date + #32 + StrPas(temp);
          TTokenizer.GetStringToken(8, #32, dirLine, temp);
            dd_Date:= dd_Date + #32 + StrPas(temp);
     { rights... }
          TTokenizer.GetStringToken(1, #32, dirLine, temp);
          dd_Rights:= StrPas(temp);
     { inodes... }
          TTokenizer.GetStringToken(2, #32, dirLine, temp);
          try
             dd_INodes:= StrToInt(StrPas(temp));
          except
             on EConvertError do dd_INodes:= 0; { no inode, no file :) }
          end;
     { owner... }
          TTokenizer.GetStringToken(3, #32, dirLine, temp);
          dd_Owner:= StrPas(temp);
     { owner group...}
          TTokenizer.GetStringToken(4, #32, dirLine, temp);
          dd_OwnerGroup:= StrPas(temp);
     { uh... now check entry type }
          case dd_Rights[1] of
               '-': dd_Info:= 'F';
               'd': dd_Info:= 'D';
               'l': dd_Info:= 'L'
          else
               dd_Info:= 'U';
          end;
     { if it's a link... }
          if dd_Info = 'L' then
          begin
          { if there's "->" link-pointer in the filename...}
             if Pos('->', dd_Name) > 0 then
             {... the link is the characters behind the "->" pointer... }
                dd_LinksTo:= Copy(dd_Name, Pos('->', dd_Name) + 3, 255)
             else
             {... we have a little problem, base, can you hear me? }
                dd_LinksTo:= 'www.microsoft.com'; { ehe... hehe... ehehehe :) }
             Delete(dd_Name, Pos('->', dd_Name), 255);  { delete all over }
          end
          else
             dd_LinksTo:= '';
     end;

{ free temporary buffer and directory info buffer }
     StrDispose(temp);
     StrDispose(dirLine);
end;
{------------------------------------------------------------------------------}
{------------------------------------------------------------------------------}
{------ constructor/destructor methods  ---------------------------------------}
{------------------------------------------------------------------------------}
{------------------------------------------------------------------------------}
constructor TDirList.Create(DirFileName: string);
begin
     inherited Create;

{ open directory info file }
     NoIOError:= true;
     AssignFile(DirFile, DirFileName);
     try
          Reset(DirFile, 1); { Record size: 1 byte }
     except
       on EInOutError do begin
            NoIOError:= false;
            raise; { reraise exception } end;
     end;

{ build table }
     LineCount:= BuildTable;
     if LineCount = 0 then
         raise EInvalidFormat.Create('Directory format doesn''t like as Unix.');
end;

{------------------------------------------------------------------------------}

destructor TDirList.Destroy;
begin
{ close directory info file }
     if NoIOError then CloseFile(DirFile);

     inherited Destroy;
end;

{------------------------------------------------------------------------------}

procedure TDirList.Free;
begin
     if Self <> nil then Destroy;
end;

end.


